/**
* \file: UspiTouchInputImpl.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Carplay - USPI Touch Adapter implementation
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
* \author: D. Devarasu /RBEI/ECF3 / dhanasekaran.d@in.bosch.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <string.h>
#include <adit_logging.h>
#include <dipo_macros.h>

#include "UspiTouchInput.h"
#include "HIDDigitizer.h"
#include "UspiTouchInputImpl.h"


LOG_IMPORT_CONTEXT(cplay_utouch);

namespace adit { namespace carplay
{

using namespace std;
using namespace uspi;

enum WaylandDeviceType
{
    AutoDetectType        = 1,
    PointerDeviceType     = 2,
    SingleTouchDeviceType = 3,
    MultiTouchDeviceType  = 4
};

UspiTouchInputImpl::UspiTouchInputImpl() : verbose(false)
{
    receiver = nullptr;
}

UspiTouchInputImpl::~UspiTouchInputImpl()
{
    deinitialize();
    receiver = nullptr;
}

bool UspiTouchInputImpl::Initialize(const IConfiguration& inConfig, IInputReceiver& inReceiver, SessionId inSessionId)
{
    receiver = &inReceiver;

    bool multiTouch = 0; // set multitouch to FALSE
    // get configuration
    int width = inConfig.GetNumber("display-width", 800);
    int height = inConfig.GetNumber("display-height", 480);
    string displayUuid = inConfig.GetItem("display-uuid", "");
    string touchUuid = inConfig.GetItem("wl-touch-uuid", "");

    int vendorId = inConfig.GetNumber("wl-touch-hid-vendor-id", 0);
    int productId = inConfig.GetNumber("wl-touch-hid-product-id", 0);
    int countryCode = inConfig.GetNumber("wl-touch-hid-country-code", 0);

    string verboseConfig;
    verbose = false;
    if (inConfig.TryGetItem("enable-verbose-logging", verboseConfig))
    {
        verbose = 0 != inConfig.GetNumber("enable-verbose-logging", 0);
        if(verbose)
            LOG_WARN((cplay_utouch, "verbose logging enabled, do not use in production!"));
    }

    // check touch-type, all other are already items checked somewhere else
    int deviceType = inConfig.GetNumber("wl-touch-type", 1);
    if (deviceType <= 0 || deviceType > 4)
    {
        LOG_WARN((cplay_utouch, "wl-touch-type: invalid configuration value. set default to 1 (auto)"));
        deviceType = AutoDetectType;
    }

    if (deviceType == AutoDetectType || deviceType == MultiTouchDeviceType)
    {
        LOG_WARN((cplay_utouch, "MultiTouch not supported in CarPlay. Change wl-touch-type in cfg file "));
        multiTouch = 1; // enable multitouch
    }

    // create HID pointer device utils
    digitizer = move(unique_ptr<HIDDigitizer>(new HIDDigitizer(touchUuid, displayUuid, "Wayland Touchscreen Device",
            multiTouch, vendorId, productId, countryCode)));

    mTouchFacade = move(unique_ptr<WaylandFacade>((new WaylandFacade(inSessionId))->
            setDisplaySize(width, height)->
            setFingerNum(1)));

    receiver->AttachInput(digitizer->AsHIDDevice());

    if(!mTouchFacade->initialize(this))
    {
        LOG_ERROR((cplay_utouch, "Failed to initialize touch facade"));
        return false;
    }

    return true;
}

void UspiTouchInputImpl::deinitialize()
{
    if (!mTouchFacade->deinitialize())
    {
        LOG_ERROR((cplay_utouch, "Failed to deinitialize touch facade"));
    }

    if (receiver != nullptr)
        receiver->DetachInput(digitizer->AsUUID());
}

void UspiTouchInputImpl::onTouch(TouchEvent inEvent)
{
    LOGD_DEBUG((cplay_utouch, "entry: type: %d, finger: %d, x: %f, y: %f",
            inEvent.eventType, inEvent.fingerId, inEvent.xRelative, inEvent.yRelative));

    switch (inEvent.eventType)
    {
    case TouchEvent::down:
    case TouchEvent::move: // motion in pressed state
    {
        HIDInputReportWithData report;
        if (digitizer->Down(inEvent.xRelative, inEvent.yRelative, inEvent.fingerId, report))
            receiver->SendInput(report.report);
        break;
    }

    case TouchEvent::up:
    {
        HIDInputReportWithData report;
        if (digitizer->Up(inEvent.fingerId, report))
            receiver->SendInput(report.report);
        break;
    }

    default:
        break;
    }
}

// todo this callback is intended to inform upper layer on an error, not log, improve usage
void UspiTouchInputImpl::onTouchError(int inError)
{
    LOG_ERROR((cplay_utouch, "%s Error = %d", __PRETTY_FUNCTION__, inError));
}

void UspiTouchInputImpl::onLogging(UspiLogLevel inLogLevel, const string& inLogString)
{
    switch (inLogLevel) {
        case USPI_LOG_FATAL:
            LOG_FATAL((cplay_utouch, "%s", inLogString.c_str()));
            break;
        case USPI_LOG_ERROR:
            LOG_ERROR((cplay_utouch, "%s", inLogString.c_str()));
            break;
        case USPI_LOG_WARN:
            LOG_WARN((cplay_utouch, "%s", inLogString.c_str()));
            break;
        case USPI_LOG_INFO:
            LOG_INFO((cplay_utouch, "%s", inLogString.c_str()));
            break;
        case USPI_LOG_DEBUG:
            LOGD_DEBUG((cplay_utouch, "%s", inLogString.c_str()));
            break;
        case USPI_LOG_VERBOSE:
            if (verbose) {
                LOGD_VERBOSE((cplay_utouch, "%s", inLogString.c_str()));
            }
            break;
        default:
            break;
    }
}

} } // namespace adit { namespace carplay
